﻿using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using VeteransAffairs.Registries.BusinessManagerAHOBPR;
using VeteransAffairs.Registries.BusinessManagerAHOBPR.Emis;
using VeteransAffairs.Registries.BusinessManagerAHOBPR.MilitaryInfoService;
using VeteransAffairs.Registries.BusinessAHOBPR;


namespace WCF_BusinessManagerAHOBPRTests
{
    [TestClass]
    public class DoDDeploymentImporterTests
    {
        private DoDDeploymentImporter _cut;

        [TestInitialize()]
        public void MyTestInitialize()
        {
            _cut = new DoDDeploymentImporter();
        }
        #region tests

        [TestMethod()]
        public void Validate_StartDate_In_CreateNewDateRange()
        {
            DateTime start = DateTime.Parse("2003-08-20");
            DateTime end = DateTime.Parse("2010-08-13");
            var ep = new MilitaryServiceEpisode();
            var data = new MilitaryServiceEpisodeData();
            data.serviceEpisodeStartDate = start;
            data.serviceEpisodeEndDate = end;
            ep.militaryServiceEpisodeData = data;
            DateRange expected = new DateRange(start, end);
            
            //DateTime actual = DateTime.Now;
            var actual = _cut.CreateNewDateRange(ep);
            Assert.AreEqual(expected.Start, actual.Start);
        }

        [TestMethod()]
        public void CreateNewDateRange_Should_Return_DeploymentEndDate_Of_11_11_9999_For_Active_Duty_Registrant()
        {
            DateTime start = DateTime.Parse("2003-08-20");
            var ep = new MilitaryServiceEpisode();
            var data = new MilitaryServiceEpisodeData();
            data.serviceEpisodeStartDate = start;
            data.serviceEpisodeEndDate = DateTime.Parse("01-01-0001");
            ep.militaryServiceEpisodeData = data;
            var expected = new DateRange(start, DateTime.Parse("11-11-9999"));

            //DateTime actual = DateTime.Now;
            var actual = _cut.CreateNewDateRange(data.serviceEpisodeStartDate, data.serviceEpisodeEndDate);
            Assert.AreEqual(expected.End, actual.End);
        }

        [TestMethod()]
        public void CreateDeployment_Should_Return_DeploymentEndDate_Of_11_11_9999_For_Registrant_With_DeploymentEndDate_Of_01_01_0001()
        {
            var location = new DeploymentLocation
            {
                deploymentLocationMajorBodyOfWaterCode = "6P",
                deploymentLocationBeginDate = DateTime.Parse("2006-01-12"),
                deploymentLocationEndDate = DateTime.Parse("0001-01-01"),
            };
            var actual = _cut.CreateDeployment("n", location);
            Assert.AreEqual(DateTime.Parse("9999-11-11").ToShortDateString(), actual.deploymentEndDate);
        }

        [TestMethod()]
        public void CreateDeployment_Will_Use_MajorBodyOfWater_For_Location_If_Present()
        {
            var location = new DeploymentLocation
            {
                deploymentLocationMajorBodyOfWaterCode = "6P",
                deploymentLocationBeginDate = DateTime.Parse("2006-01-12"),
                deploymentLocationEndDate = DateTime.Parse("2006-05-13"),
            };
            var actual = _cut.CreateDeployment("n", location);
            Assert.AreEqual("6P",actual.deploymentCountryCode);
        }

        [TestMethod()]
        public void CreateDeployment_Should_Use_DeploymentCountry_For_Location_If_Present()
        {
            var location = new DeploymentLocation
            {
                deploymentCountry = "IZ",
                deploymentLocationBeginDate = DateTime.Parse("2006-01-12"),
                deploymentLocationEndDate = DateTime.Parse("2006-05-13"),
            };
            var actual = _cut.CreateDeployment("n", location);
            Assert.AreEqual("IZ", actual.deploymentCountryCode);
        }

       

        [TestMethod()]
        public void Validate_EndDate_In_CreateNewDateRange()
        {
            DateTime start = DateTime.Parse("2003-08-20");
            DateTime end = DateTime.Parse("2010-08-13");
            MilitaryServiceEpisode ep = new MilitaryServiceEpisode();
            MilitaryServiceEpisodeData data = new MilitaryServiceEpisodeData();
            data.serviceEpisodeStartDate = start;
            data.serviceEpisodeEndDate = end;
            ep.militaryServiceEpisodeData = data;
            DateRange expected = new DateRange(start, end);
            
            //DateTime actual = DateTime.Now;
            var actual = _cut.CreateNewDateRange(ep);
            Assert.AreEqual(expected.End, actual.End);
        }

        [TestMethod()]
        public void GetBranchOfService_With_Army_BoS_Episode()
        {
            var deploymentStart = DateTime.Parse("2003-08-20");
            var deploymentEnd = DateTime.Parse("2010-08-13");
            Dictionary<DateRange, string> svcEpisodes = CreateSvcDateRange();
            var actual = _cut.GetBranchOfService(deploymentStart, deploymentEnd, svcEpisodes);
            Assert.AreEqual("Army", actual);
        }

        [TestMethod()]
        public void GetBranchOfService_With_Invalid_BoSEpisode_Should_Return_99()
        {
            var deploymentStart = DateTime.Parse("1776-07-04");
            var deploymentEnd = DateTime.Parse("1781-10-19");
            //Surrender at Yorktown. In case you're wondering
            var svcEpisodeStart = DateTime.Parse("2003-08-20");
            var svcEpisodeEnd = DateTime.Parse("2010-08-13");
            var svcDateRange = new DateRange(svcEpisodeStart, svcEpisodeEnd);
            Dictionary<DateRange, string> svcEpisodes = new Dictionary<DateRange, string> { { svcDateRange, "A" } };
            
            string actual = _cut.GetBranchOfService(deploymentStart, deploymentEnd, svcEpisodes);
            Assert.AreEqual("Unknown", actual);
        }

        [TestMethod()]
        public void AssociateBranchesOfServiceWithDatesServed_Test()
        {
            DateRange expectedKey = new DateRange(DateTime.Parse("2003-08-20"), DateTime.Parse("2010-08-13"));
            var edipi = "6001010004";
            var inputEdipi = new InputEdipiIcn();
            inputEdipi.edipiORicnValue = edipi;
            inputEdipi.inputType = InputEdipiIcnInputType.EDIPI;
            var input = new inputEdiPiOrIcn();
            input.edipiORicn = inputEdipi;
            
            var inputHeaderInfo = new InputHeaderInfo();
            List<MilitaryServiceEpisode> list = _cut.GetMilitaryServiceEpisodes(edipi);
            Dictionary<DateRange, string> actual = _cut.AssociateBranchesOfServiceWithDatesServed(list);
            
           Assert.AreEqual(expectedKey.Start, actual.Keys.First().Start, "Failure: Expected Start value: " + expectedKey.Start + " does not equal actual Start value: " + actual.Keys.First().Start);
            Assert.AreEqual(expectedKey.End, actual.Keys.First().End, "Failure: Expected End value: " + expectedKey.End + " does not equal actual End value: " + actual.Keys.First().End);
        }

        [TestMethod()]
        public void AssociateBranchesOfServiceWithDatesServed_Is_Not_Null_When_Given_Empty_List()
        {
            
            var actual = _cut.AssociateBranchesOfServiceWithDatesServed(new List<MilitaryServiceEpisode>());
            Assert.IsNotNull(actual);
            Assert.AreEqual(0, actual.Keys.Count);
        }

        [TestMethod()]
        public void GetMilitaryServiceEpisodes_Response_Should_Have_One_Entry()
        {
            var edipi = "6001010004";
            var inputEdipi = new InputEdipiIcn();
            inputEdipi.edipiORicnValue = edipi;
            inputEdipi.inputType = InputEdipiIcnInputType.EDIPI;
            var input = new inputEdiPiOrIcn();
            input.edipiORicn = inputEdipi;
            
            var actual = _cut.GetMilitaryServiceEpisodesResponse(new InputHeaderInfo(), input);
            Assert.AreEqual(1, actual.militaryServiceEpisode.Length);
        }

        [TestMethod()]
        public void GetMilitaryServiceEpisodes_Returns_An_Empty_List_When_No_ServiceEpisodes_Exist()
        {
            var edipi = "6001010112";
            
            var actual = _cut.GetMilitaryServiceEpisodes(edipi);
            Assert.IsNotNull(actual);
            Assert.AreEqual(0, actual.Count);
        }

        [TestMethod()]
        public void GetMilitaryServiceEpisodes_WhenNoServiceEpisodes_Exist_Returned_List_Is_Not_Null()
        {
            var invalidedipi = "6001010112";
            
            var actual = _cut.GetMilitaryServiceEpisodes(invalidedipi);
            Assert.IsNotNull(actual);
        }

        [TestMethod()]
        public void GetMilitaryServiceEpisodes_Returns_A_List()
        {
            var edipi = "6001010004";
            var inputEdipi = new InputEdipiIcn();
            inputEdipi.edipiORicnValue = edipi;
            inputEdipi.inputType = InputEdipiIcnInputType.EDIPI;
            var input = new inputEdiPiOrIcn();
            input.edipiORicn = inputEdipi;
            
            var inputHeaderInfo = new InputHeaderInfo();
            List<MilitaryServiceEpisode> actual = _cut.GetMilitaryServiceEpisodes(edipi);
            Assert.AreEqual(1, actual.Count);
        }

        [TestMethod()]
        public void SetRegistrantDeployments_Test()
        {
            Dictionary<DateRange, string> svcDateRange = CreateSvcDateRange();
            eMISdeploymentResponseType response = new eMISdeploymentResponseType
            {
                deployment = CreateDeploymentTestData()
            };           
            BprDoDDeployment[] actual = _cut.SetRegistrantDeployments(response, svcDateRange);
            Assert.IsNotNull(actual);
            Assert.AreEqual(2, actual.Length);

        }

        [TestMethod()]
        public void SetRegistrantDeployments_Should_Continue_To_Process_Deployments_If_There_Is_A_Null_Deployment_Location()
        {
            Dictionary<DateRange, string> svcDateRange = CreateSvcDateRange();
            Deployment[] deployments = new Deployment[2];
            Deployment depOne = new Deployment
            {
                deploymentData = new DeploymentData
                {
                    DeploymentLocation = CreateDeploymentLocations(),
                    deploymentStartDate = DateTime.Parse("2005-11-01"),
                    deploymentEndDate = DateTime.Parse("2006-11-29")
                }
            };
            deployments[0] = depOne;
            Deployment deploymentWithNoLocations = new Deployment
            {
                deploymentData = new DeploymentData
                {
                    deploymentProjectCode = "9GF",
                    deploymentSegmentIdentifier = "1",
                    deploymentTerminationReason = "S"
                }
            };
            deployments[1] = deploymentWithNoLocations;

            eMISdeploymentResponseType response = new eMISdeploymentResponseType
            {
                deployment = deployments
            };
            BprDoDDeployment[] actual = _cut.SetRegistrantDeployments(response, svcDateRange);
            Assert.IsNotNull(actual);
            Assert.AreEqual(2, actual.Length);

        }

        [TestMethod()]
        public void CreateDeployment_Should_Not_Be_Null_When_Given_Valid_Data()
        {
            DeploymentLocation loc = CreateDeploymentLocation("KU", "2006-07-29", "2006-07-30");
            
            BprDoDDeployment actual = _cut.CreateDeployment("a", loc);
            Assert.IsNotNull(actual);
        }

        [TestMethod()]
        public void CreateDeployment_Should_Process_A_Deployment_With_No_End_Date()
        {
            DeploymentLocation loc = CreateDeploymentLocation("KU", "2006-07-29", "2006-07-30");

            BprDoDDeployment actual = _cut.CreateDeployment("a", loc);
            Assert.IsNotNull(actual);
        }

        [TestMethod()]
        [ExpectedException(typeof(ArgumentNullException))]
        public void Create_Deployment_Throws_ArgumentNullException_When_DeploymentLocation_Is_Null()
        {
            
            BprDoDDeployment actual = _cut.CreateDeployment("a", null);
            Assert.IsNotNull(actual);
        }

        [TestMethod()]
        public void Create_Deployment_Should_Have_Correct_DeploymentLocation_Info()
        {
            DeploymentLocation loc = CreateDeploymentLocation("KU", "2006-07-29", "2006-07-30");
            
            BprDoDDeployment actual = _cut.CreateDeployment("a", loc);
            Assert.AreEqual("KU", actual.deploymentCountryCode);
            Assert.AreEqual("a", actual.branchOfService);
            Assert.AreEqual("7/30/2006",actual.deploymentEndDate);
            Assert.AreEqual("7/29/2006", actual.deploymentStartDate);
        }

        [TestMethod()]
        public void TypeTest()
        {
            var assembly = Assembly.GetAssembly(typeof(AHOBPRBaseBO));
            var importer = new BprDoDDeploymentImporter();
            Type type = assembly.GetType("VeteransAffairs.Registries.BusinessManagerAHOBPR.BprDoDDeploymentImporter");
            IDeploymentImporter actual = Activator.CreateInstance(type) as IDeploymentImporter;
            Assert.IsInstanceOfType(actual, typeof(BprDoDDeploymentImporter));
        }

#endregion
        #region SampleData

        public Deployment[] CreateDeploymentTestData()
        {
            Deployment depOne = new Deployment
            {
                deploymentData = new DeploymentData
                {
                    DeploymentLocation = CreateDeploymentLocations(),
                    deploymentStartDate = DateTime.Parse("2005-11-01"),
                    deploymentEndDate = DateTime.Parse("2006-11-29")
                }
            };
            return new Deployment[] { depOne };
        }

        private  DeploymentLocation CreateDeploymentLocation(string country, string beginDate, string endDate)
        {
            return new DeploymentLocation
            {
                deploymentCountry = country, 
                deploymentLocationBeginDate = DateTime.Parse(beginDate),
                deploymentLocationEndDate =  DateTime.Parse(endDate),
            };
        }
        private  DeploymentLocation[] CreateDeploymentLocations()
        {
            var locOne = CreateDeploymentLocation("KU", "2006-07-29", "2006-07-30");

            var locTwo = new DeploymentLocation
            {
                deploymentCountry = "IZ",
                deploymentLocationBeginDate = DateTime.Parse("2006-01-12"),
                deploymentLocationEndDate = DateTime.Parse("2006-05-13"),
            };
            return new DeploymentLocation[] { locOne, locTwo };
        }
        private Dictionary<DateRange, string> CreateSvcDateRange()
        {
            var svcEpisodeStart = DateTime.Parse("2003-08-20");
            var svcEpisodeEnd = DateTime.Parse("2010-08-13");
            var svcDateRange = new DateRange(svcEpisodeStart, svcEpisodeEnd);
            Dictionary<DateRange, string> svcEpisodes = new Dictionary<DateRange, string> { { svcDateRange, "A" } };
            return svcEpisodes;
        }
        #endregion
    }
}
